import { useState, useRef } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import { BlockControls } from '@wordpress/block-editor';
import { Popover, ToolbarGroup, DropdownMenu, TextControl, TextareaControl, SelectControl } from '@wordpress/components';
import { insert, registerFormatType } from '@wordpress/rich-text';
import { useSelect } from '@wordpress/data';
import './index.scss';

const name = 'docy/docy-toolbar';

const Docy_Toolbar = ({ value, onChange }) => {
	const [showPopover, setShowPopover] = useState(false);
	const [selectedText, setSelectedText] = useState('');
	const [activeButton, setActiveButton] = useState(null);
	const [tooltipContent, setTooltipContent] = useState('');
	const [tooltipId, setTooltipId] = useState('');
	const [selectedTaxonomy, setSelectedTaxonomy] = useState('');

	const buttonRef = useRef(null);

	// Fetch available taxonomies
	const taxonomies = useSelect((select) => {
		const tax = select('core').getTaxonomies();
		if (!tax) return [];
		return tax.map((taxonomy) => ({
			label: taxonomy.name,
			value: taxonomy.slug,
		}));
	}, []);

	const handleToolbarClick = (buttonKey) => {
		setActiveButton(buttonKey);
		const selectedRange = value.text.slice(value.start, value.end);
		setSelectedText(selectedRange || '');

		if (buttonKey === 'tooltip') {
			setTooltipId('tooltip-' + Math.floor(Math.random() * 10000));
			setTooltipContent('');
		}

		if (buttonKey === 'terms') {
			setSelectedTaxonomy('');
		}

		setShowPopover(true);
	};

	const insertShortcode = () => {
		let formattedText = selectedText.trim();
		if (!formattedText) {
			formattedText = __('Empty content', 'docy-core');
		}

		let shortcode = '';

		switch (activeButton) {
			case 'direction':
				shortcode = `[direction]${formattedText}[/direction]`;
				break;

			case 'tooltip':
				shortcode = `[docy_tooltip id="${tooltipId}" content="${tooltipContent}" trigger="hover"]${formattedText}[/docy_tooltip]`;
				break;

			case 'terms':
				if (!selectedTaxonomy) return;
				shortcode = `[terms tax="${selectedTaxonomy}"][/terms]`;
				break;

			default:
				shortcode = formattedText;
		}

		const newValue = insert(value, shortcode);
		onChange(newValue);
		setShowPopover(false);
	};

	return (
		<>
			<BlockControls>
				<ToolbarGroup>
					<div ref={buttonRef}>
						<DropdownMenu
							className='docy-toolbar__dropdown'
							icon="docy-prev-icon"
							label={__('Insert Docy Shortcode', 'docy-core')}
							controls={[
								{
									icon: 'editor-code',
									title: __('Direction', 'docy-core'),
									onClick: () => handleToolbarClick('direction'),
								},
								{
									icon: 'editor-code',
									title: __('Tooltip', 'docy-core'),
									onClick: () => handleToolbarClick('tooltip'),
								},
								{
									icon: 'editor-code',
									title: __('Terms', 'docy-core'),
									onClick: () => handleToolbarClick('terms'),
								}
							]}
						/>
					</div>
				</ToolbarGroup>
			</BlockControls>

			{showPopover && (
				<Popover
					anchorRef={buttonRef.current}
					position="bottom center"
					onClose={() => setShowPopover(false)}
					className="docy-direction-dropdown-tool"
				>
					<div className="direction-shortcode-wrap">
						{activeButton === 'direction' && (
							<>
                                <strong>{__('Insert Direction Shortcode', 'docy-core')}</strong>

                                <p className="direction-description">
                                    {__('Use this tool to format step-by-step directions in your content.', 'docy-core')}
                                </p>

                                <div className="direction-fields">
                                    <TextControl
                                        help={__('Enter steps: Step 1 > Step 2 > Step 3', 'docy-core')}
                                        value={selectedText}
                                        onChange={(value) => setSelectedText(value)}
                                    />
                                    <span className='submit-btn' onClick={insertShortcode}></span>
                                </div>
							</>
						)}

						{activeButton === 'tooltip' && (
							<>
								<strong>{__('Insert Tooltip Shortcode', 'docy-core')}</strong>
								<TextControl
									label={__('Text to Show', 'docy-core')}
									value={selectedText}
									onChange={(value) => setSelectedText(value)}
								/>
								<TextareaControl
									label={__('Tooltip Content', 'docy-core')}
									value={tooltipContent}
									onChange={(value) => setTooltipContent(value)}
								/>                                
                                <button className='submit-btn button button-primary' onClick={insertShortcode}>
                                    {__('Insert Shortcode', 'docy-core')}
                                </button>
							</>
						)}

						{activeButton === 'terms' && (
							<>
								<strong>{__('Insert Terms Shortcode', 'docy-core')}</strong>
								<SelectControl
									label={__('Select Taxonomy', 'docy-core')}
									value={selectedTaxonomy}
									options={[
										{ label: __('Select a taxonomy', 'docy-core'), value: '' },
										...taxonomies,
									]}
									onChange={(value) => setSelectedTaxonomy(value)}
								/>
                                <button className='submit-btn button button-primary' onClick={insertShortcode}>                                    
                                    {__('Insert Shortcode', 'docy-core')}
                                </button>
							</>
						)}

					</div>
				</Popover>
			)}
		</>
	);
};

registerFormatType(name, {
	title: __('Docy Toolbar', 'docy-core'),
	tagName: 'span',
	className: 'docy-toolbar',
	edit: Docy_Toolbar,
});