<?php
/**
 * Plugin Name: Docy Core
 * Requires Plugins: elementor
 * Plugin URI: https://themeforest.net/user/spider-themes/portfolio
 * Description: This plugin adds the core features to the Docy WordPress theme. You must have to install this plugin to get all the features included with the Docy theme.
 * Version: 4.1.4
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 8.0
 * Author: spider-themes
 * Author URI: https://themeforest.net/user/spider-themes/portfolio
 * Text domain: docy-core
 * Domain Path: /languages
 * License: GPL2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
	die( '-1' );
}

// Docy Core Directories
define( 'DC_IMAGES', plugins_url( 'widgets/images/', __FILE__ ) );

require_once __DIR__ . '/vendor/autoload.php';
define( 'DOCY_PATH', plugin_dir_path( __FILE__ ) );
define( 'DOCY_URL', plugin_dir_url( __FILE__ ) );


// Make sure the same class is not loaded twice in free/premium versions.
if ( ! class_exists( 'Docy_core' ) ) {
	/**
	 * Main Docy Core Class
	 *
	 * The main class that initiates and runs the Docy Core plugin.
	 */
	class Docy_core {

		/**
		 * Docy Core Version
		 *
		 * Holds the version of the plugin.
		 *
		 * @var string The plugin version.
		 */
		const VERSION = '4.1.4';

		/**
		 * Instance
		 *
		 * Holds a single instance of the `Docy_Core` class.
		 *
		 * @access private
		 * @static
		 *
		 * @var Docy_Core A single instance of the class.
		 */
		private static $_instance = null;

		/**
		 * Instance
		 *
		 * Ensures only one instance of the class is loaded or can be loaded.
		 *
		 * @access public
		 * @static
		 *
		 * @return Docy_Core An instance of the class.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}

			return self::$_instance;
		}

		/**
		 * Clone
		 *
		 * Disable class cloning.
		 *
		 * @access protected
		 *
		 * @return void
		 */
		public function __clone() {
			// Cloning instances of the class is forbidden
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Cheatin&#8217; huh?', 'docy-core' ), '1.7.0' );
		}

		/**
		 * Wakeup
		 *
		 * Disable unserializing the class.
		 *
		 * @access protected
		 *
		 * @return void
		 */
		public function __wakeup() {
			// Unserializing instances of the class is forbidden.
			_doing_it_wrong( __FUNCTION__, esc_html__( 'Cheatin&#8217; huh?', 'docy-core' ), '1.7.0' );
		}

		/**
		 * Constructor
		 *
		 * Initialize the Docy Core plugins.
		 *
		 * @access public
		 */
		public function __construct() {
			$this->init_hooks();
			$this->core_includes();
			do_action( 'docy_core_loaded' );
			add_action( 'init', [ $this, 'load_acf_files' ] );
		}

		/**
		 * Load ACF files
		 *
		 * Load ACF related files.
		 *
		 * @access public
		 */
		public function load_acf_files() {

			// ACF Meta Box
			require_once __DIR__ . '/post-type/acf_meta.php';

			// Gutenberg Blocks
			if ( function_exists( 'acf_register_block_type' ) ) {
				require_once __DIR__ . '/blocks/_blocks.php';
			}

			// Blocks
			require_once __DIR__ . '/inc/blocks.php';
		}

		/**
		 * Include Files
		 *
		 * Load core files required to run the plugin.
		 *
		 * @access public
		 */
		public function core_includes(): void {
			// Get option values from theme options

			// Extra functions
			require_once __DIR__ . '/inc/extra.php';

			// Custom post types
			$opt               = get_option( 'docy_opt' );
			$post_type_disable = $opt['post_type_disable'] ?? '';

			// get faq from array
			if ( is_array( $post_type_disable ) ) {
				in_array( 'faq', $post_type_disable ) ? require_once __DIR__ . '/post-type/faq.pt.php' : '';
				in_array( 'video', $post_type_disable ) ? require_once __DIR__ . '/post-type/video.pt.php' : '';
			}

			//Header Footer
			require_once __DIR__ . '/post-type/Header_Footer.pt.php';
			new DocyCore\PostType\Header_Footer();

			// Elementor custom section
			require DOCY_PATH . 'widgets/ElementorAdvance/Advance_Section.php';
			new DocyCore\widgets\ElementorAdvance\Advance_Section();

			/**
			 * Register widget area.
			 *
			 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
			 */
			require_once __DIR__ . '/wp-widgets/widgets.php';

			// Elementor custom field icons
			require_once __DIR__ . '/inc/icons.php';

			// Shortcodes
			require_once __DIR__ . '/shortcodes/code.php';
			require_once __DIR__ . '/shortcodes/direction.php';
			require_once __DIR__ . '/shortcodes/tooltip.php';
			require_once __DIR__ . '/shortcodes/tags.php';
            require_once __DIR__ . '/shortcodes/login-form.php';
            require_once __DIR__ . '/shortcodes/share-post.php';
		}

		/**
		 * Init Hooks
		 *
		 * Hook into actions and filters.
		 *
		 * @access private
		 */
		private function init_hooks(): void {
			add_action( 'plugins_loaded', [ $this, 'init_plugin' ] );
		}

		/**
		 * Init Docy Core
		 *
		 * Load the plugin after Elementor (and other plugins) are loaded.
		 *
		 * @access public
		 */
		public function init_plugin(): void {

			add_filter( 'elementor/icons_manager/additional_tabs', [ $this, 'font_icons' ] );

			// Add new Elementor Categories
			add_action( 'elementor/init', [ $this, 'add_elementor_category' ] );

			// Register Widget Scripts
			add_action( 'elementor/frontend/after_register_scripts', [ $this, 'register_widget_scripts' ] );
			add_action( 'elementor/editor/before_enqueue_scripts', [ $this, 'register_widget_scripts' ] );
			add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );

			// Register Widget Scripts
			add_action( 'elementor/editor/before_enqueue_scripts', [ $this, 'enqueue_elementor_editor_styles' ] );
			add_action( 'elementor/frontend/after_enqueue_styles', [ $this, 'enqueue_widget_styles' ] );

			// Editor Styles
			add_action( 'enqueue_block_editor_assets', [ $this, 'gutenberg_editor_scripts' ] );

			add_action( 'admin_enqueue_scripts', [ $this, 'register_admin_styles' ] );

			// Register New Widgets
			add_action( 'elementor/widgets/register', [ $this, 'on_widgets_registered' ] );

			add_filter( 'wpml_elementor_widgets_to_translate', [ $this, 'wpml_widgets_to_translate_filter' ] );

			// Elementor Block & Page Template Library
			if ( did_action( 'elementor/loaded' ) ) {
				require_once __DIR__ . '/inc/template_library/Template_Library.php';
				new DocyCore\inc\template_library\Template_Library();
			}
		}

		/**
		 * Integrate WPML
		 */
		public function wpml_widgets_to_translate_filter( $widgets ) {
			require_once __DIR__ . '/wpml/WPML_Fields.php';

			return $widgets;
		}

		/**
		 * Add new Elementor Categories
		 *
		 * Register new widget categories for Docy Core widgets.
		 *
		 * @since  1.0.0
		 * @since  1.7.1 The method moved to this class.
		 *
		 * @access public
		 */
		public function add_elementor_category(): void {
			\Elementor\Plugin::instance()->elements_manager->add_category( 'docy-elements', [
				'title' => esc_html__( 'Docy Elements', 'docy-core' ),
			], 1 );
		}

		/**
		 * Register Widget Styles
		 *
		 * Register custom styles required to run Docy Core.
		 *
		 * @access public
		 */
		public function enqueue_widget_styles(): void {
			wp_register_style( 'prism', plugins_url( 'assets/vendors/prism/prism.min.css', __FILE__ ) );
			wp_register_style( 'nice-select', plugins_url( 'assets/vendors/niceselectpicker/nice-select.css', __FILE__ ) );
			wp_register_style( 'simple-line-icon', plugins_url( 'assets/vendors/simple-line-icon/simple-line-icons.css', __FILE__ ) );
			wp_register_style( 'animated', plugins_url( 'assets/vendors/slick/animated.css', __FILE__ ) );
			wp_register_style( 'slick', plugins_url( 'assets/vendors/slick/slick.css', __FILE__ ) );
			wp_register_style( 'slick-theme', plugins_url( 'assets/vendors/slick/slick-theme.css', __FILE__ ) );
			wp_register_style( 'elegant-icon', plugins_url( 'assets/vendors/elegant-icon/style.css', __FILE__ ) );
			wp_register_style( 'video-js', plugins_url( 'assets/vendors/video/videojs.min.css', __FILE__ ) );
			wp_register_style( 'video-js-theaterMode', plugins_url( 'assets/vendors/video/videojs.theaterMode.css', __FILE__ ) );
		}

		/**
		 * Register Widget Scripts
		 *
		 * Register custom scripts required to run Docy Core.
		 *
		 * @access public
		 */
		public function register_widget_scripts(): void {
			wp_register_script( 'ajax-chimp', plugins_url( 'assets/js/ajax-chimp.js', __FILE__ ), 'jquery', '1.0' );
			wp_register_script( 'mixitup', plugins_url( 'assets/js/mixitup.min.js', __FILE__ ), 'jquery', '1.0' );
			wp_register_script( 'prism', plugins_url( 'assets/vendors/prism/prism.js', __FILE__ ), array( 'jquery' ), '1.17.1' );
			wp_register_script( 'tooltipster', plugins_url( 'assets/vendors/tooltipster/js/tooltipster.bundle.min.js', __FILE__ ), array( 'jquery' ), '4.2.8' );
			wp_register_script( 'parallaxie', plugins_url( 'assets/js/parallaxie.js', __FILE__ ), array( 'jquery' ), '0.5' );
			wp_register_script( 'nice-select', plugins_url( 'assets/vendors/niceselectpicker/jquery.nice-select.min.js', __FILE__ ), 'jquery', '1.0' );
			wp_register_script( 'floatThead', plugins_url( 'assets/js/jquery.floatThead.min.js', __FILE__ ), array( 'jquery' ), '2.1.4' );
			wp_register_script( 'counterup', plugins_url( 'assets/vendors/counterup/jquery.counterup.min.js', __FILE__ ), array( 'jquery' ), '1.0.0' );
			wp_register_script( 'waypoints', plugins_url( 'assets/vendors/counterup/jquery.waypoints.min.js', __FILE__ ), array( 'jquery' ), '4.0.1' );
			wp_register_script( 'tweenmax', plugins_url( 'assets/js/TweenMax.min.js', __FILE__ ), array( 'jquery' ), '2.0.0' );
			wp_register_script( 'wavify', plugins_url( 'assets/js/jquery.wavify.js', __FILE__ ), array( 'jquery' ), '1.0.0' );
			wp_register_script( 'chart', plugins_url( 'assets/js/Chart.js', __FILE__ ), array( 'jquery' ), '1.0.0' );
			wp_register_script( 'slick', plugins_url( 'assets/vendors/slick/slick.min.js', __FILE__ ), array( 'jquery' ), '1.9.0' );
			wp_register_script( 'artplayer', plugins_url( 'assets/vendors/video/artplayer.js', __FILE__ ), array( 'jquery' ), '3.5.26' );
			wp_register_script( 'mcustomscrollbar', plugins_url( 'assets/vendors/mcustomscrollbar/jquery.mCustomScrollbar.concat.min.js', __FILE__ ), array( 'jquery' ), '3.1.13' );
			wp_register_script( 'parallax-scroll', plugins_url( 'assets/js/jquery.parallax-scroll.js', __FILE__ ), array( 'jquery' ), '3.1.13' );
			wp_register_script( 'video-js', plugins_url( 'assets/vendors/video/video.min.js', __FILE__ ), array( 'jquery' ), '7.6.0' );
			wp_register_script( 'video-js-nuevo', plugins_url( 'assets/vendors/video/nuevo.min.js', __FILE__ ), array( 'jquery' ), '7.6.0' );
			wp_register_script( 'scripts', plugins_url( 'assets/js/scripts.js', __FILE__ ), array( 'jquery' ), '3.6.0' );
		}

		public function enqueue_elementor_editor_styles(): void {
			wp_enqueue_style( 'docy-elementor-editor', plugins_url( 'assets/css/elementor-editor.css', __FILE__ ) );
		}

		public function enqueue_scripts(): void {
			wp_deregister_style( 'elementor-animations' );
			wp_deregister_style( 'e-animations' );

			if ( is_post_type_archive( 'topic' ) ) {
				wp_deregister_style( 'bbp-default' );
			}
		}

		public function register_admin_styles(): void {
			wp_enqueue_style( 'docy-gutenberg-editor', plugins_url( 'assets/css/gutenberg-editor.css', __FILE__ ) );
		}

		public function gutenberg_editor_scripts(): void {
			wp_enqueue_style( 'docy-gutenberg-editor', plugins_url( 'assets/css/gutenberg-editor.css', __FILE__ ) );
		}

		/**
		 * Register New Widgets
		 *
		 * Include Docy Core widgets files and register them in Elementor.
		 *
		 * @access public
		 */
		public function on_widgets_registered(): void {
			$this->include_widgets();
			$this->register_elementor_widgets();
		}

		/***
		 * Added Custom Font Icon Integrated Elementor Icon Library
		 */
		public function font_icons( $custom_fonts ) {
			$css_data  = plugins_url( 'assets/vendors/elegant-icon/style.css', __FILE__ );
			$json_data = plugins_url( '/assets/vendors/elegant-icon/eleganticons.json', __FILE__ );

			$custom_fonts['elegant-icon'] = [
				'name'          => 'elegant-icon',
				'label'         => esc_html__( 'Elegant Icons', 'docy' ),
				'url'           => $css_data,
				'prefix'        => '',
				'displayPrefix' => '',
				'labelIcon'     => 'icon_star',
				'ver'           => '',
				'fetchJson'     => $json_data,
				'native'        => true,
			];

			return $custom_fonts;
		}

		/**
		 * Include Widgets Files
		 *
		 * Load Docy Core widgets files.
		 *
		 * @access private
		 */
		private function include_widgets(): void {
			require_once __DIR__ . '/vendor/autoload.php';
		}

		/**
		 * Register Widgets
		 *
		 * Register Docy Core widgets.
		 *
		 * @access private
		 */
		private function register_elementor_widgets(): void {
			// bbPress widgets
			$forum_widgets = [];
			if ( class_exists( 'bbPress' ) ) {
				$forum_widgets = [ 'Forums', 'Forum_topics', 'Single_forum' ];
			}

			$post_type_disable = function_exists( 'docy_opt' ) ? docy_opt( 'post_type_disable' ) : '';
			$faq       = [];
			$video     = [];

			if ( is_array( $post_type_disable ) ) {
				$faq       = in_array( 'faq', $post_type_disable ) ? [ 'Faq' ] : [];
				$video     = in_array( 'video', $post_type_disable ) ? [ 'Video_playlist' ] : [];
			}

			$widgets = [
				'Hero',
				'Docs',
				'Accordion',
				'Button',
				'Testimonial',
				'Subscribe',
				'Tabs',
				'Process_Tabs',
				'Alerts_box',
				'Data_Table',
				'Image_HotSpots',
				'Counter',
				'Quote',
				'Info_box',
				'Cheatsheet',
				'List_item',
				'Call_to_action',
				'Search_Form',
				'Accordion_Article',
				'Nav_Menu',
			];

			$widgets = array_merge( $widgets, $forum_widgets, $faq, $video );

			foreach ( $widgets as $widget ) {
				$classname = "\\DocyCore\\Widgets\\$widget";
				if ( function_exists( 'Docy_helper' ) ) {
					\Elementor\Plugin::instance()->widgets_manager->register( new $classname() );
				}
			}
		}
	}
}

if ( ! function_exists( 'docy_core_load' ) ) {
	/**
	 * Load Docy Core
	 *
	 * Main instance of Docy_Core.
	 *
	 */
	function docy_core_load() {
		return Docy_core::instance();
	}

	// Run Docy Core
	docy_core_load();
}