<?php
namespace SPEL_PRO\Widgets;

use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Stack image Widgets
 */
class Stacked_Image extends Widget_Base
{

    // Enqueue Lightbox assets when lightbox is enabled
    public function enqueue_lightbox_assets(): void
    {
        // Check if Lightbox is enabled in the widget settings
        if ( 'yes' === $this->get_settings_for_display( 'enable_lightbox' ) ) {
            wp_enqueue_script( 'elementor-frontend' ); // Ensure Elementor's frontend JS is loaded
            wp_enqueue_style( 'elementor-lightbox' );  // Ensure Elementor's lightbox CSS is loaded
        }
    }

    // Hook to enqueue the lightbox assets
    public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_lightbox_assets' ] );  // Enqueue on the frontend
    }

    public function get_name (): string
    {
        return 'spel_stacked_image'; // ID of the widget (Don't change this name)
    }

    public function get_title (): string
    {
        return esc_html__('Stacked Image', 'spider-elements-pro');
    }

    public function get_icon (): string
    {
        return 'eicon-featured-image spel-icon';
    }

    public function get_keywords (): array
    {
        return [ 'spider', 'Image', 'stacked image' ];
    }

    public function get_categories (): array
    {
        return [ 'spider-element-pro' ];
    }

    /**
	 * Name: get_style_depends()
	 * Desc: Register the required CSS dependencies for the frontend.
	 */
	public function get_style_depends(): array
    {
		return [ 'spel-main-pro' ];
	}

	/**
	 * Name: get_script_depends()
	 * Desc: Register the required JS dependencies for the frontend.
	 */
	public function get_script_depends(): array
    {
		return ['spel-el-pro-widgets' ];
	}

    /**
     * Name: register_controls()
     * Desc: Register controls for these widgets
     * Params: no params
     * Return: @void
     * Since: @1.0.0
     * Package: @spider-elements
     * Author: spider-themes
     */
    protected function register_controls (): void
    {
        $this->elementor_content_control();
        $this->elementor_style_control();

    }

    /**
     * Name: elementor_content_control()
     * Desc: Register the Content Tab output on the Elementor editor.
     * Params: no params
     * Return: @void
     * Since: @1.0.0
     * Package: @spider-elements
     * Author: spider-themes
     */
    public function elementor_content_control (): void
    {

        //===================== Select Preset ===========================//
        $this->start_controls_section(
            'style_sec', [
                'label' => esc_html__('Preset Skins', 'spider-elements-pro'),
            ]
            );
            
        $this->add_control(
            'style', [
                'label' => esc_html__('Style', 'spider-elements-pro'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    '1' => [
                        'icon' => 'stack_img_1',
                        'title' => esc_html__('01 : Default style', 'spider-elements-pro')
                    ],
                    '2' => [
                        'icon' => 'stack_img_2',
                        'title' => esc_html__('02 : Hover style', 'spider-elements-pro'),
                    ],
                    '3' => [
                        'icon' => 'stack_img_3',
                        'title' => esc_html__('03 : Hover style', 'spider-elements-pro'),
                    ],
                    '4' => [
                        'icon' => 'stack_img_4',
                        'title' => esc_html__('04 : Scrolling style', 'spider-elements-pro'),
                    ],
                ],
                'default' => '1',
            ]
        );

        $this->end_controls_section(); //End Select Style

        $this->start_controls_section(
            'stack_images',
            [
                'label' => esc_html__('Stack Image Gallery', 'spider-elements-pro'),
            ]
        );
        $this->add_control(
            'stack_image',
            [
                'label' => esc_html__( 'Add Images', 'spider-elements-pro' ),
                'type' => Controls_Manager::GALLERY,

            ]
        );

         // Add Lightbox switcher control
         $this->add_control(
            'enable_lightbox',
            [
                'label' => __( 'Lightbox', 'spider-elements-pro' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __( 'Yes', 'spider-elements-pro' ),
                'label_off' => __( 'No', 'spider-elements-pro' ),
                'default' => 'yes', // Default is enabled
                'condition' => [
                    'style' => '3',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Name: elementor_style_control()
     * Desc: Register the Style Tab output on the Elementor editor.
     * Params: no params
     * Return: @void
     * Since: @1.0.0
     * Package: @spider-elements
     * Author: spider-themes
     */
    public function elementor_style_control (): void
    {
        $this->start_controls_section(
            'section_image_style',
            [
                'label' => esc_html__('Image Style', 'spider-elements-pro'),
                'tab' => Controls_Manager::TAB_STYLE,
	            'condition' => [
		            'style' => ['1', '2', '3']
	            ]
            ]
        );
        $this->add_control(
            'stack_image_alignment',
            [
                'label' => esc_html__('Image Alignment', 'spider-elements-pro'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'spider-elements-pro'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'top' => [
                        'title' => esc_html__('Center', 'spider-elements-pro'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'spider-elements-pro'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'left',
                'toggle' => true,
            ]
        );

	    $this->add_group_control(
		    Group_Control_Background::get_type(),
		    [
			    'name' => 'stack_background',
			    'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
			    'selector' => '{{WRAPPER}} .stack_image,
			                   {{WRAPPER}} .stack_image_container',
			    'condition' => [
				    'style' => ['1', '2', '4']
			    ]
		    ]
	    );

        $this->add_responsive_control(
            'stack_image_width',
            [
                'label' => esc_html__('Width', 'spider-elements-pro'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ '%', 'px' ],
				'separator' => 'before',
                'desktop_default' => [
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'range' => [
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 50,
                        'max' => 1000,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .stack_image' => 'width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .imgstack .stack_img' => 'width: {{SIZE}}{{UNIT}};',
                ],
	            'condition' => [
					'style' => ['1', '3', '4']
	            ]
            ]
        );

        $this->add_responsive_control(
            'stack_image_height',
            [
                'label' => esc_html__('Height', 'spider-elements-pro'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 50,
                        'max' => 1000,
                    ],
                ],
                'selectors' => [
	                '{{WRAPPER}} .stack_image' => 'height: {{SIZE}}{{UNIT}};',
	                '{{WRAPPER}} .stack_image_container' => 'height: {{SIZE}}{{UNIT}};',
	                '{{WRAPPER}} .imgstack .stack_img' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'hotspot_image_border',
                'selector' => '{{WRAPPER}} .stack_image,
                               {{WRAPPER}} .stack_image_container,
                               {{WRAPPER}} .imgstack .stack_img',
	            'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'stack_image_border_radius',
            [
                'label' => esc_html__('Border Radius', 'spider-elements-pro'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
				'separator' => 'after',
                'selectors' => [
                    '{{WRAPPER}} .spel_stack_image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .stack_image_container' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .imgstack .stack_img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

	    $this->add_group_control(
		    Group_Control_Box_Shadow::get_type(), [
			    'name'     => 'stack_image_hover_box_shadow',
			    'selector' => '{{WRAPPER}} .spel_stack_image,
			                   {{WRAPPER}} .stack_image_container,
			                   {{WRAPPER}} .imgstack .stack_img',
		    ]
	    );

        $this->end_controls_section();

    }


    /**
     * Name: elementor_render()
     * Desc: Render the widget output on the frontend.
     * Params: no params
     * Return: @void
     * Since: @1.0.0
     * Package: @spider-elements
     * Author: spider-themes
     */
    protected function render (): void
    {
        $settings = $this->get_settings_for_display();
        extract($settings); //extract all settings array to variables converted to name of key

        //Include template parts
        include "templates/stack-image/stack-image-{$settings['style']}.php";
    }

}


