/* eslint-env amd */
/* globals module:false */

// https://github.com/umdjs/umd/blob/master/templates/returnExports.js
(function (root, factory) {
    'use strict';

    if (typeof define === 'function' && define.amd) {
        // AMD. Register as an anonymous module.
        define([], factory);
    } else if (typeof module === 'object' && module.exports) {
        // Node. Does not work with strict CommonJS, but
        // only CommonJS-like environments that support module.exports,
        // like Node.
        module.exports = factory();
    } else {
        // Browser globals (root is window)
        root.AnchorJS = factory();
        root.anchors = new root.AnchorJS();
    }
}(this, function () {
    'use strict';

    function AnchorJS(options) {
        this.options = options || {};
        this.elements = [];

        /**
         * Assigns options to the internal options object, and provides defaults.
         * @param {Object} opts - Options object
         */
        function _applyRemainingDefaultOptions(opts) {
            opts.icon = opts.hasOwnProperty('icon') ? opts.icon : '\ue9cb'; // Accepts characters (and also URLs?), like  '#', '¶', '❡', or '§'.
            opts.visible = opts.hasOwnProperty('visible') ? opts.visible : 'hover'; // Also accepts 'always' & 'touch'
            opts.placement = opts.hasOwnProperty('placement') ? opts.placement : 'right'; // Also accepts 'left'
            opts.ariaLabel = opts.hasOwnProperty('ariaLabel') ? opts.ariaLabel : 'Anchor'; // Accepts any text.
            opts.class = opts.hasOwnProperty('class') ? opts.class : ''; // Accepts any class name.
            opts.base = opts.hasOwnProperty('base') ? opts.base : ''; // Accepts any base URI.
            // Using Math.floor here will ensure the value is Number-cast and an integer.
            opts.truncate = opts.hasOwnProperty('truncate') ? Math.floor(opts.truncate) : 64; // Accepts any value that can be typecast to a number.
            opts.titleText = opts.hasOwnProperty('titleText') ? opts.titleText : ''; // Accepts any text.
        }

        _applyRemainingDefaultOptions(this.options);

        /**
         * Checks to see if this device supports touch. Uses criteria pulled from Modernizr:
         * https://github.com/Modernizr/Modernizr/blob/da22eb27631fc4957f67607fe6042e85c0a84656/feature-detects/touchevents.js#L40
         * @return {Boolean} - true if the current device supports touch.
         */
        this.isTouchDevice = function () {
            return !!(('ontouchstart' in window) || window.DocumentTouch && document instanceof DocumentTouch);
        };

        /**
         * Add anchor links to page elements.
         * @param  {String|Array|Nodelist} selector - A CSS selector for targeting the elements you wish to add anchor links
         *                                            to. Also accepts an array or nodeList containing the relavant elements.
         * @return {this}                           - The AnchorJS object
         */
        this.add = function (selector) {
            var elements,
                elsWithIds,
                idList,
                elementID,
                i,
                index,
                count,
                tidyText,
                newTidyText,
                anchor,
                visibleOptionToUse,
                hrefBase,
                indexesToDrop = [];

            // We reapply options here because somebody may have overwritten the default options object when setting options.
            // For example, this overwrites all options but visible:
            //
            // anchors.options = { visible: 'always'; }
            _applyRemainingDefaultOptions(this.options);

            visibleOptionToUse = this.options.visible;
            if (visibleOptionToUse === 'touch') {
                visibleOptionToUse = this.isTouchDevice() ? 'always' : 'hover';
            }

            // Provide a sensible default selector, if none is given.
            if (!selector) {
                selector = 'h2, h3, h4, h5, h6';
            }

            elements = _getElements(selector);

            if (elements.length === 0) {
                return this;
            }

            _addBaselineStyles();

            // We produce a list of existing IDs so we don't generate a duplicate.
            elsWithIds = document.querySelectorAll('[id]');
            idList = [].map.call(elsWithIds, function assign(el) {
                return el.id;
            });

            for (i = 0; i < elements.length; i++) {
                if (this.hasAnchorJSLink(elements[i])) {
                    indexesToDrop.push(i);
                    continue;
                }

                if (elements[i].hasAttribute('id')) {
                    elementID = elements[i].getAttribute('id');
                } else if (elements[i].hasAttribute('data-anchor-id')) {
                    elementID = elements[i].getAttribute('data-anchor-id');
                } else {
                    tidyText = this.urlify(elements[i].textContent);

                    // Compare our generated ID to existing IDs (and increment it if needed)
                    // before we add it to the page.
                    newTidyText = tidyText;
                    count = 0;
                    do {
                        if (index !== undefined) {
                            newTidyText = tidyText + '-' + count;
                        }

                        index = idList.indexOf(newTidyText);
                        count += 1;
                    } while (index !== -1);
                    index = undefined;
                    idList.push(newTidyText);

                    elements[i].setAttribute('id', newTidyText);
                    elementID = newTidyText;
                }

                // The following code efficiently builds this DOM structure:
                // `<a class="anchorjs-link ${this.options.class}"
                //     aria-label="${this.options.ariaLabel}"
                //     data-anchorjs-icon="${this.options.icon}"
                //     title="${this.options.titleText}"
                //     href="this.options.base#${elementID}">
                // </a>;`
                anchor = document.createElement('a');
                anchor.className = 'anchorjs-link ' + this.options.class;
                anchor.setAttribute('aria-label', this.options.ariaLabel);
                anchor.setAttribute('tabindex', '-1');
                anchor.setAttribute('data-anchorjs-icon', this.options.icon);
                if (this.options.titleText) {
                    anchor.title = this.options.titleText;
                }
                // Adjust the href if there's a <base> tag. See https://github.com/bryanbraun/anchorjs/issues/98
                hrefBase = document.querySelector('base') ? window.location.pathname + window.location.search : '';
                hrefBase = this.options.base || hrefBase;
                anchor.href = hrefBase + '#' + elementID;

                if (visibleOptionToUse === 'always') {
                    anchor.style.opacity = '1';
                }

                if (this.options.icon === '\ue9cb') {
                    anchor.style.font = '1em/1 anchorjs-icons';

                    // We set lineHeight = 1 here because the `anchorjs-icons` font family could otherwise affect the
                    // height of the heading. This isn't the case for icons with `placement: left`, so we restore
                    // line-height: inherit in that case, ensuring they remain positioned correctly. For more info,
                    // see https://github.com/bryanbraun/anchorjs/issues/39.
                    if (this.options.placement === 'left') {
                        anchor.style.lineHeight = 'inherit';
                    }
                }

                if (this.options.placement === 'left') {
                    elements[i].insertBefore(anchor, elements[i].firstChild);
                } else { // if the option provided is `right` (or anything else).
                    elements[i].appendChild(anchor);
                }
            }

            for (i = 0; i < indexesToDrop.length; i++) {
                elements.splice(indexesToDrop[i] - i, 1);
            }
            this.elements = this.elements.concat(elements);

            return this;
        };

        /**
         * Removes all anchorjs-links from elements targeted by the selector.
         * @param  {String|Array|Nodelist} selector - A CSS selector string targeting elements with anchor links,
         *                                            OR a nodeList / array containing the DOM elements.
         * @return {this}                           - The AnchorJS object
         */
        this.remove = function (selector) {
            var index,
                domAnchor,
                elements = _getElements(selector);

            for (var i = 0; i < elements.length; i++) {
                domAnchor = elements[i].querySelector('.anchorjs-link');
                if (domAnchor) {
                    // Drop the element from our main list, if it's in there.
                    index = this.elements.indexOf(elements[i]);
                    if (index !== -1) {
                        this.elements.splice(index, 1);
                    }
                    // Remove the anchor from the DOM.
                    elements[i].removeChild(domAnchor);
                }
            }
            return this;
        };

        /**
         * Removes all anchorjs links. Mostly used for tests.
         */
        this.removeAll = function () {
            this.remove(this.elements);
        };

        /**
         * Urlify - Refine text so it makes a good ID.
         *
         * To do this, we remove apostrophes, replace non-safe characters with hyphens,
         * remove extra hyphens, truncate, trim hyphens, and make lowercase.
         *
         * @param  {String} text - Any text. Usually pulled from the webpage element we are linking to.
         * @return {String}      - hyphen-delimited text for use in IDs and URLs.
         */
        this.urlify = function (text) {
            // Regex for finding the non-safe URL characters (many need escaping): & +$,:;=?@"#{}|^~[`%!'<>]./()*\ (newlines, tabs, backspace, & vertical tabs)
            var nonsafeChars = /[& +$,:;=?@"#{}|^~[`%!'<>\]\.\/\(\)\*\\\n\t\b\v]/g,
                urlText;

            // The reason we include this _applyRemainingDefaultOptions is so urlify can be called independently,
            // even after setting options. This can be useful for tests or other applications.
            if (!this.options.truncate) {
                _applyRemainingDefaultOptions(this.options);
            }

            // Note: we trim hyphens after truncating because truncating can cause dangling hyphens.
            // Example string:                      // " ⚡⚡ Don't forget: URL fragments should be i18n-friendly, hyphenated, short, and clean."
            urlText = text.trim()                   // "⚡⚡ Don't forget: URL fragments should be i18n-friendly, hyphenated, short, and clean."
                .replace(/\'/gi, '')                  // "⚡⚡ Dont forget: URL fragments should be i18n-friendly, hyphenated, short, and clean."
                .replace(nonsafeChars, '-')           // "⚡⚡-Dont-forget--URL-fragments-should-be-i18n-friendly--hyphenated--short--and-clean-"
                .replace(/-{2,}/g, '-')               // "⚡⚡-Dont-forget-URL-fragments-should-be-i18n-friendly-hyphenated-short-and-clean-"
                .substring(0, this.options.truncate)  // "⚡⚡-Dont-forget-URL-fragments-should-be-i18n-friendly-hyphenated-"
                .replace(/^-+|-+$/gm, '')             // "⚡⚡-Dont-forget-URL-fragments-should-be-i18n-friendly-hyphenated"
                .toLowerCase();                       // "⚡⚡-dont-forget-url-fragments-should-be-i18n-friendly-hyphenated"

            return urlText;
        };

        /**
         * Determines if this element already has an AnchorJS link on it.
         * Uses this technique: http://stackoverflow.com/a/5898748/1154642
         * @param    {HTMLElement}  el - a DOM node
         * @return   {Boolean}     true/false
         */
        this.hasAnchorJSLink = function (el) {
            var hasLeftAnchor = el.firstChild && ((' ' + el.firstChild.className + ' ').indexOf(' anchorjs-link ') > -1),
                hasRightAnchor = el.lastChild && ((' ' + el.lastChild.className + ' ').indexOf(' anchorjs-link ') > -1);

            return hasLeftAnchor || hasRightAnchor || false;
        };

        /**
         * Turns a selector, nodeList, or array of elements into an array of elements (so we can use array methods).
         * It also throws errors on any other inputs. Used to handle inputs to .add and .remove.
         * @param  {String|Array|Nodelist} input - A CSS selector string targeting elements with anchor links,
         *                                         OR a nodeList / array containing the DOM elements.
         * @return {Array} - An array containing the elements we want.
         */
        function _getElements(input) {
            var elements;
            if (typeof input === 'string' || input instanceof String) {
                // See https://davidwalsh.name/nodelist-array for the technique transforming nodeList -> Array.
                elements = [].slice.call(document.querySelectorAll(input));
                // I checked the 'input instanceof NodeList' test in IE9 and modern browsers and it worked for me.
            } else if (Array.isArray(input) || input instanceof NodeList) {
                elements = [].slice.call(input);
            } else {
                throw new Error('The selector provided to AnchorJS was invalid.');
            }
            return elements;
        }

        /**
         * _addBaselineStyles
         * Adds baseline styles to the page, used by all AnchorJS links irregardless of configuration.
         */
        function _addBaselineStyles() {
            // We don't want to add global baseline styles if they've been added before.
            if (document.head.querySelector('style.anchorjs') !== null) {
                return;
            }

            var style = document.createElement('style'),
                linkRule =
                    ' .anchorjs-link {' +
                    '   opacity: 0;' +
                    '   text-decoration: none;' +
                    '   -webkit-font-smoothing: antialiased;' +
                    '   -moz-font-smoothing: grayscale;' +
                    ' }',
                hoverRule =
                    ' *:hover > .anchorjs-link,' +
                    ' .anchorjs-link:focus  {' +
                    '   opacity: 1;' +
                    ' }',
                anchorjsLinkFontFace =
                    ' @font-face {' +
                    '   font-family: "anchorjs-icons";' + // Icon from icomoon; 10px wide & 10px tall; 2 empty below & 4 above
                    '   src: url(data:n/a;base64,AAEAAAALAIAAAwAwT1MvMg8yG2cAAAE4AAAAYGNtYXDp3gC3AAABpAAAAExnYXNwAAAAEAAAA9wAAAAIZ2x5ZlQCcfwAAAH4AAABCGhlYWQHFvHyAAAAvAAAADZoaGVhBnACFwAAAPQAAAAkaG10eASAADEAAAGYAAAADGxvY2EACACEAAAB8AAAAAhtYXhwAAYAVwAAARgAAAAgbmFtZQGOH9cAAAMAAAAAunBvc3QAAwAAAAADvAAAACAAAQAAAAEAAHzE2p9fDzz1AAkEAAAAAADRecUWAAAAANQA6R8AAAAAAoACwAAAAAgAAgAAAAAAAAABAAADwP/AAAACgAAA/9MCrQABAAAAAAAAAAAAAAAAAAAAAwABAAAAAwBVAAIAAAAAAAIAAAAAAAAAAAAAAAAAAAAAAAMCQAGQAAUAAAKZAswAAACPApkCzAAAAesAMwEJAAAAAAAAAAAAAAAAAAAAARAAAAAAAAAAAAAAAAAAAAAAQAAg//0DwP/AAEADwABAAAAAAQAAAAAAAAAAAAAAIAAAAAAAAAIAAAACgAAxAAAAAwAAAAMAAAAcAAEAAwAAABwAAwABAAAAHAAEADAAAAAIAAgAAgAAACDpy//9//8AAAAg6cv//f///+EWNwADAAEAAAAAAAAAAAAAAAAACACEAAEAAAAAAAAAAAAAAAAxAAACAAQARAKAAsAAKwBUAAABIiYnJjQ3NzY2MzIWFxYUBwcGIicmNDc3NjQnJiYjIgYHBwYUFxYUBwYGIwciJicmNDc3NjIXFhQHBwYUFxYWMzI2Nzc2NCcmNDc2MhcWFAcHBgYjARQGDAUtLXoWOR8fORYtLTgKGwoKCjgaGg0gEhIgDXoaGgkJBQwHdR85Fi0tOAobCgoKOBoaDSASEiANehoaCQkKGwotLXoWOR8BMwUFLYEuehYXFxYugC44CQkKGwo4GkoaDQ0NDXoaShoKGwoFBe8XFi6ALjgJCQobCjgaShoNDQ0NehpKGgobCgoKLYEuehYXAAAADACWAAEAAAAAAAEACAAAAAEAAAAAAAIAAwAIAAEAAAAAAAMACAAAAAEAAAAAAAQACAAAAAEAAAAAAAUAAQALAAEAAAAAAAYACAAAAAMAAQQJAAEAEAAMAAMAAQQJAAIABgAcAAMAAQQJAAMAEAAMAAMAAQQJAAQAEAAMAAMAAQQJAAUAAgAiAAMAAQQJAAYAEAAMYW5jaG9yanM0MDBAAGEAbgBjAGgAbwByAGoAcwA0ADAAMABAAAAAAwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABAAH//wAP) format("truetype");' +
                    ' }',
                pseudoElContent =
                    ' [data-anchorjs-icon]::after {' +
                    '   content: attr(data-anchorjs-icon);' +
                    ' }',
                firstStyleEl;

            style.className = 'anchorjs';
            style.appendChild(document.createTextNode('')); // Necessary for Webkit.

            // We place it in the head with the other style tags, if possible, so as to
            // not look out of place. We insert before the others so these styles can be
            // overridden if necessary.
            firstStyleEl = document.head.querySelector('[rel="stylesheet"], style');
            if (firstStyleEl === undefined) {
                document.head.appendChild(style);
            } else {
                document.head.insertBefore(style, firstStyleEl);
            }

            style.sheet.insertRule(linkRule, style.sheet.cssRules.length);
            style.sheet.insertRule(hoverRule, style.sheet.cssRules.length);
            style.sheet.insertRule(pseudoElContent, style.sheet.cssRules.length);
            style.sheet.insertRule(anchorjsLinkFontFace, style.sheet.cssRules.length);
        }
    }

    return AnchorJS;
}));


